#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <float.h>
#include <math.h>
#include <time.h>

#include "pepAlignGA.h"

// Defining some unix/linux standard functions for windows
#ifdef WIN32
void srand48(int seed)
{
	srand(seed);

}// void srand48(int seed)


double drand48()
{
	return (double)((double)rand())/((double)(RAND_MAX));

}// double drand48()
#endif

static double floatmax(double l, double r) {
	if (l >= r)
		return l;
	return r;
}

static double floatmin(double l, double r) {
	if (l >= r)
		return r;
	return l;
}

static int intmin(int l, int r) {
	if (l <= r)
		return l;
	return r;
}

// Some local global variables
static double xmin;		// maximum x_value
double xmax;			// maximum x_value
static double ymin;		// minimum y_value
double ymax;			// maximum y_value
static double x_spread;
static double y_spread;


// Calculates fitness
double calc_fitness(double** data_points, int npoints, pplf_type plf, double lc_sigma)
{
	double f=0, k, m, denum, num, divres;
	int bpi, ppi;	// breakpoint index and plotpoint index
	int counted = 0, outofbounds=0, partcount;

    denum = lc_sigma*lc_sigma; 

	// checking out of bounds members
	ppi = 0;
	while ((data_points[ppi][0] < plf->xpoints[0])) {
		outofbounds++;
		ppi++;
	}// if

	for(bpi=0; bpi<plf->nof_points-1; bpi++) {	
		
		/* define y=k*x+m for line segment between breakpoints */
		partcount = 0;				
		while ( ppi < npoints && (data_points[ppi][0] >= plf->xpoints[bpi]) && (data_points[ppi][0] < plf->xpoints[bpi+1])) { 
			k = (plf->ypoints[bpi+1] - plf->ypoints[bpi]) / (plf->xpoints[bpi+1] - plf->xpoints[bpi]);
			m = plf->ypoints[bpi] - (k * plf->xpoints[bpi]);
			num = pow((data_points[ppi][1]-(k*data_points[ppi][0]+m)), 2);
			divres = exp(-(num/denum));
			f += divres;											
			partcount++;
			ppi++;
		}/* if */
		
		if (partcount == 0)
			f -= 100;
		else
			counted += partcount;
	}// for
    
	f = (f - (0.5 * plf->nof_points));				/* cost per breakpoint */
	f = (f - ((npoints-counted)+ outofbounds));		/* cost per missed point */

	return f;

}// double fitness(double A[MAX_DATAPOINTS][2], int nA, PLF_type S)


static int comp_func(const void *a, const void *b)
{
	double lv = *(double*)a, rv = *(double*)b;
	
	if (lv >= 0) {
		if (rv >= 0) {
			if ((lv - rv) > 0)
				return 1;
			else if ((lv - rv) < 0)
				return -1;
			else
				return 0;
		}// if		
		else
			return -1;
	}// if
	else {
		if (rv >= 0) 
			return 1;
		else
			return 0;
	}// else  

}// static int comp_func(const void *a, const void *b)


/* Procedure that initialises the GA and related structures*/
void init_alignment(pplf_type plf_array, int size, double** data_points, int n_points)
{
	int i, j;
	
	srand48((unsigned)time(0));
	/* srand48(22069); */

	// Initializing the boundaries
	xmin = DBL_MAX;
	xmax = 0;
	ymin = DBL_MAX;
	ymax = 0;

	for (i=0; i<n_points; i++)  {
		if (data_points[i][0] < xmin)
			xmin = data_points[i][0];
		else if (data_points[i][0] > xmax)
			xmax = data_points[i][0];
		if (data_points[i][1] < ymin)
			ymin = data_points[i][1];
		else if (data_points[i][1] > ymax)
			ymax = data_points[i][1];
	}// for

	// Creating a bit of space to breathe 
	xmin = xmin*0.95;
	xmax = xmax*1.05;
	ymin = ymin*0.95;
	ymax = ymax*1.05;

	x_spread = xmax - xmin;
	y_spread = ymax - ymin;

	// Initializing the workspace by random
	for(i=0; i<size; i++) {
		plf_array[i].xpoints[0] = 0;
		plf_array[i].ypoints[0] = 0;
		for(j=1; j<((MAX_BREAKPOINTS)/2)+1; j++) {
			plf_array[i].xpoints[j] = xmin + drand48()*x_spread;
			plf_array[i].ypoints[j] = ymin + drand48()*y_spread;
	    }// for
		plf_array[i].xpoints[j] = xmax;
		plf_array[i].ypoints[j] = ymax;

		for(j+=1; j<MAX_BREAKPOINTS; j++) {
			plf_array[i].xpoints[j] = -1;
			plf_array[i].ypoints[j] = -1;
	    }// for

		plf_array[i].nof_points = ((MAX_BREAKPOINTS)/2)+1; 
		plf_array[i].fitness = -100;
		sort_breakpoints(&(plf_array[i]));
	}// for

}// void init_alignment(pplf_type plf_array, int size, double data_points[MAX_DATAPOINTS][2], int n_points)


/* Procedure that sorts the breakpoints in a plf_type in ascending order */
void sort_breakpoints(pplf_type plf)
{
	int ipoint;

	qsort(plf->xpoints, MAX_BREAKPOINTS, sizeof(double), comp_func);
	qsort(plf->ypoints, MAX_BREAKPOINTS, sizeof(double), comp_func);

	/* Adapting the number of points */
	plf->nof_points = MAX_BREAKPOINTS;
	for(ipoint=MAX_BREAKPOINTS-1; ipoint>=0; ipoint--) {
		if (plf->xpoints[ipoint] == -1 || plf->ypoints[ipoint] == -1)
			plf->nof_points -= 1;
		else {
			break;
		}// else
	}/* for */
	
}/* void sort_breakpoints(pplf_type plf) */


/* Function that copies the content of one candidate to another */
void copy_candidate(pplf_type dst, pplf_type src)
{
	int i;

	for(i=0; i<MAX_BREAKPOINTS; i++) {
		dst->xpoints[i] = src->xpoints[i];
		dst->ypoints[i] = src->ypoints[i];
	}/* for */
	
	dst->nof_points = src->nof_points;
	dst->fitness = src->fitness;
			
}/* void copy_candidate(pplf_type dst, pplf_type src) */


static int cand_comp_func(const void *a, const void *b)
{	 
	plf_type lv = *(plf_type*)a;
	plf_type rv = *(plf_type*)b;

	if (rv.fitness > lv.fitness)
		return 1;
	else if (rv.fitness < lv.fitness)
		return -1;
	else
		return 0;

}// static int cand_comp_func(const void *a, const void *b)

/* Function that sorts the PLF array in descending order according to their fitness */
void sort_candidates(pplf_type plf_array, int size)
{
	qsort(plf_array, size, sizeof(plf_type), cand_comp_func);

}/* void sort_candidates(pplf_type plf_array, int size) */


/* Function that runs a selection and crossover procedure on the srcarray  */
void select_cross_over(pplf_type src_array, int size, double selection_ratio)
{
	int s_cutoff;
	int i, pos;	
	int ubound, lbound;
	int parent, parent2;

	/* CrossOver step, make combinations of the first selection_ratio parents
	 * to fill the remaining 1-selection_ratio options
	 */
	s_cutoff = floor((selection_ratio*size)+0.5); 
	for(i=s_cutoff; i<size-1; i+=2)
    {
		/* Choosing first parent to use in a crossover */
		parent = floor((drand48()*(s_cutoff-1))+0.5);
		parent2 = floor((drand48()*(s_cutoff-1))+0.5);
		while (parent2 == parent)
			parent2 = floor((drand48()*(s_cutoff-1))+0.5);

		/* Copying the parent to the new candidate */
		copy_candidate(&src_array[i], &src_array[parent]);
		copy_candidate(&src_array[i+1], &src_array[parent2]);

		lbound = 1;
		ubound = intmin(src_array[i].nof_points, src_array[i+1].nof_points);		
		pos = lbound + floor((drand48()*(ubound-lbound)));
		perform_crossover(&src_array[i], &src_array[i+1], pos);		
	}/* for */
}


/* Function that detemines a suitable position for a crossover operation */
void perform_crossover(pplf_type child1, pplf_type child2, int pos)
{
	int j;
	double temp;

	/* If the startposition is at the beginning or end of the array deny crossover */
	if (pos < 1 || pos >child2->nof_points-1 || pos >child1->nof_points-1)
		return;

	for (j=1; j<pos; j++) {
		temp = child2->xpoints[j]; child2->xpoints[j] = child1->xpoints[j]; child1->xpoints[j] = temp;
		temp = child2->ypoints[j]; child2->ypoints[j] = child1->ypoints[j];	child1->ypoints[j] = temp;
	}// for		

	sort_breakpoints(child1);
	sort_breakpoints(child2);
	child1->fitness = -100;
	child2->fitness = -100;
	
}/* int perform_crossover(pplf_type parent1, pplf_type parent2, pplf_type child, int startpos) */


/* Mutates the designated samples with a certain random distribution */
void mutate_samples(pplf_type src_array, int size, double selection_ratio)
{
	static int gen = 1;
	int mod = 0;
	int s_cutoff;
	int i, j;
	
	/* Mutate step, mutate the first selection_ratio parents
	 * to fill the remaining 1-selection_ratio options
	 */
	s_cutoff = floor((selection_ratio*size)+0.5);

	for(i=s_cutoff; i<size; i++)
    {	    
		mod = 0;
		for(j=1; j<MAX_BREAKPOINTS; j++) {			
			// the endpoint is protected
			if (j== src_array[i].nof_points-1)
				continue;

			if(src_array[i].xpoints[j] > 0 && drand48() < 0.5*(gen/N_GENERATIONS)) {					
				src_array[i].xpoints[j] += 10*(drand48()-0.5); /* nudge X */
				src_array[i].ypoints[j] +=  3*(drand48()-0.5); /* nudge Y */
				src_array[i].xpoints[j] = floatmax(src_array[i].xpoints[j], xmin);
				src_array[i].ypoints[j] = floatmax(src_array[i].ypoints[j], ymin);
				mod = 1;
			}// else if

			else if(drand48() < 0.05) { /* major mutation/insertion */ 	
				src_array[i].xpoints[j] = xmin + drand48()*(x_spread);
				src_array[i].ypoints[j] = ymin + drand48()*(y_spread);
				mod = 1;
			}// if
		  
			else if(src_array[i].xpoints[j] > 0 && drand48() < 0.05) { /* delete breakpoint */				
					src_array[i].xpoints[j] = -1;
					src_array[i].ypoints[j] = -1;
					mod = 1;				
			}/* else if */		
		}// for
				
		if (mod == 1) {										
			sort_breakpoints(&(src_array[i]));
		}
	}/* for */

	gen++;

}/* void mutate_samples(pplf_type src_array, int size, double selection_ratio) */


/* Function that performs Genetic Algorithm alignment */
pplf_type run_ga_alignment(double** data_points, int n_points, double lc_sigma, char* output_base, double* x_max, double* y_max)
{
	char temp_output[1000];
	plf_type alignments[N_CANDIDATES];	/* alignment vector */
	pplf_type retval;					/* return value */
	int i, j;							/* loop index */
	FILE* fout;							/* output file for the progress plots */

	/* Preparing output file */
	strcpy(temp_output, output_base);
	strcat(temp_output, STD_GTF_EXT);
	fout = fopen(temp_output, "w");

	// Printing header
	fprintf(fout, "#Generation");
	for (i=0; i<CURVES_DISPLAYED; i++)
		fprintf(fout, "\tCurve %i fitness", i+1);
	fprintf(fout, "\n");

	// Performing random initialization
	init_alignment(alignments, N_CANDIDATES, data_points, n_points);
	
	/* Calculate fitness and sort in decreasing order */
	for(j=0; j<N_CANDIDATES; j++) {
		alignments[j].fitness = calc_fitness(data_points, n_points, &(alignments[j]), lc_sigma);
	}/* for */
	sort_candidates(alignments, N_CANDIDATES);

	/* Loop through N_GENERATIONS generations */
	for(i=0; i<N_GENERATIONS; i++) 
	{	  
		/* CrossOver step */
		select_cross_over(alignments, N_CANDIDATES, FRACTION_KEPT);

		/* Mutation step */
		mutate_samples(alignments, N_CANDIDATES, FRACTION_KEPT);

		/* Calculate fitness and sort in decreasing order */
		for(j=floor((FRACTION_KEPT*N_CANDIDATES)+0.5); j<N_CANDIDATES; j++) {
			alignments[j].fitness = calc_fitness(data_points, n_points, &(alignments[j]), lc_sigma);
	    }/* for */
		sort_candidates(alignments, N_CANDIDATES);

		/* Plot the top CURVES_DISPLAYED for the progress file */
		fprintf(fout, "%i", i);
		for (j=0; j<CURVES_DISPLAYED; j++)
			fprintf(fout, "\t%f", alignments[j].fitness);
		fprintf(fout, "\n");
	}/* for */

	fclose(fout);

	retval = (pplf_type) malloc(sizeof(plf_type));
	if (!retval) {
		printf("Error: Memory allocation for the ga_value failed.\n"); fflush(stdout);
	}// if
	copy_candidate(retval, &(alignments[0]));
	*x_max = xmax;
	*y_max = ymax;

	return retval;
			
}/* pplf_type run_ga_alignment(double data_points[MAX_DATAPOINTS][2], int npoints, double x_max, double y_max, double lc_sigma) */
