#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "pepAlignFunctions.h"
#include "pepAlignDefs.h"
#include "pepXMLReader.h"
#include "mzXMLReader.h"
#include "PeptideProphetDelegate.h"

/* Generates a gnuplot compatible script that plots the training training progress */
void gen_train_plot(parameters* params)
{
	char tempname[1000];
	FILE* fout;
	int i; 	
	strcpy(tempname, params->output_file_base);
	strcat(tempname, ".train");
	strcat(tempname, STD_PLT_EXT);
	fout = fopen(tempname, "w");

	fprintf(fout, "set autoscale\n");
	fprintf(fout, "unset log\n");
	fprintf(fout, "unset label\n");
	fprintf(fout, "set xtic auto\n");
	fprintf(fout, "set ytic auto\n");
	fprintf(fout, "set title \"Alignment training progress\"\n");
	fprintf(fout, "set xlabel \"Generation\"\n");
	fprintf(fout, "set ylabel \"Score\"\n");		
	
	strcpy(tempname, params->output_file_base);
	strcat(tempname, STD_GTF_EXT);

	fprintf(fout, "plot ");
	for (i=1; i<=CURVES_DISPLAYED; i++) {
		fprintf(fout, "'%s' using 1:%i title 'Result %i %1' with lines lt %i", tempname, i+1, i, i);
		if (i != CURVES_DISPLAYED)
			fprintf(fout, ", \\");
		fprintf(fout, "\n");
	}// for
	fclose(fout);
		
}

/* Generates a gnuplot compatible script that plots the training files and the alignment */
void gen_align_plot(parameters* params)
{
	char tempname[1000];
	FILE* fout;

	strcpy(tempname, params->output_file_base);
	strcat(tempname, ".align");
	strcat(tempname, STD_PLT_EXT);
	fout = fopen(tempname, "w");

	fprintf(fout, "set autoscale\n");
	fprintf(fout, "unset log\n");
	fprintf(fout, "unset label\n");
	fprintf(fout, "set xrange [0:]\n");
	fprintf(fout, "set yrange [0:]\n");
	fprintf(fout, "set xtic auto\n");
	fprintf(fout, "set ytic auto\n");

	if (params->alignmode) {
		fprintf(fout, "set title \"Retention time alignment of MS data with MS/MS data\"\n");
		fprintf(fout, "set xlabel \"MS/MS retention time\"\n");
		fprintf(fout, "set ylabel \"MS retention time\"\n");		
	}// if
	else {
		fprintf(fout, "set title \"Scan alignment of MS data with MS/MS data\"\n");
		fprintf(fout, "set xlabel \"MS/MS scan number\"\n");
		fprintf(fout, "set ylabel \"MS scan number\"\n");
	}// else

	strcpy(tempname, params->output_file_base);
	strcat(tempname, STD_DSF_EXT);
	fprintf(fout, "plot \'%s\'", tempname);
	if (params->alignmode) 
		fprintf(fout, "using 5:7 notitle with points lt -1, \\\n");
	else
		fprintf(fout, "using 4:6 notitle with points lt -1, \\\n");

	strcpy(tempname, params->output_file_base);
	strcat(tempname, STD_PLF_EXT);
	fprintf(fout, "\'%s\' using 1:2 notitle with linespoints lt 3 lw 1.5", tempname);
	fclose(fout);

}// void gen_align_plot(parameters* params)

void show_help()
{	
	printf("usage: pepAlign -p<MS/MS pepXML filename> -m<LC-MS data in mzXML> -e<max. mass error in MS-only data (in ppm)> -t<Selected score of the MS/MS petides> [-l<typical standard deviation in LC retention time in LC-MS data> -P<min,max MS/MS scan for alignment)> -R<min, max MS scan for alignment)> -o<output file> -L<lowerbound on selected score> -U<Upperbound on selected score> -M<align mode, either scan or time>] (type pepAlign --help for more information)\n");

}// void show_help()


int main(int argc, char* argv[])
{
	char temp_output[1000];
	parameters* pars;
	pmsms_pipeline_analysis pepxml_file;	
	pdelegate_list dlgl;
	pdelegate_type dlg; 
	pmzxml_file mzxml_file;
	peptide_type* peps;
	double** best_point;	
	int num_points;
	int pepnum, i;
	FILE* fout;

	// Reading parameters
	printf("Reading parameters... "); fflush(stdout);
	pars = pepalign_filter_parameters(argc, argv); 
	if (!pars) {
		show_help();
		return -1;
	}

	printf("done.\nReading pepXML file %s... ", pars->pepxml_file); fflush(stdout);
	// Reading pepXML_file, enabling peptideprohet 
	dlgl = make_delegate_list();
	dlg = make_peptide_prophet_result_delegate();
	add_delegate_to_list(dlgl, dlg);
	pepxml_file = read_pepxml_file(pars->pepxml_file, 0, 0, dlgl);	

	if (!pepxml_file) {
		printf("Error: pepXML file was not loaded\n");
		return -1;
	}// if
	
	printf("done.\nFiltering peptides... "); fflush(stdout);
	// Filtering out peptides and making them unique
	peps = read_peptides(pepxml_file, pars, &pepnum);
	unify_peptide_array(peps, &pepnum, pars);

	printf("done.\nReading mzXML file %s... ", pars->mzxml_file); fflush(stdout);
	// Reading mzXML file
	mzxml_file = read_mzxml_file_spectrum(pars->mzxml_file, 0, 0, pars->ms_start_scan, pars->ms_end_scan);      

	if (!mzxml_file) {
		printf("Error: mzXML file was not loaded\n");
		return -1;
	}// if

	if (mzxml_file->scan_num < pars->ms_end_scan)
		pars->ms_end_scan = mzxml_file->scan_num;

	printf("done.\nAligning files... "); fflush(stdout);
	// Running the alignment
	best_point = align_peptides(mzxml_file, peps, pepnum, pars, &num_points);

	printf("done.\nWriting alignment... "); fflush(stdout);
	// Outputting the best points to the filename		
	strcpy(temp_output, pars->output_file_base);
	strcat(temp_output, STD_PLF_EXT);
	fout = fopen(temp_output, "w");

	// Printing mode
	fprintf(fout, "#mode %i\n", pars->alignmode); fflush(fout);
	
	// Printing header
	fprintf(fout, "#x_coordinate\ty_coordinate\n"); fflush(fout);

	for (i=0; i<num_points; i++) {
		if (best_point[i][0] > -1 && best_point[i][1] > -1)
			fprintf(fout, "%f\t%f\n", best_point[i][0], best_point[i][1]); fflush(fout);
	}// for

	fclose(fout);
	printf("done.\n"); fflush(stdout);

	gen_align_plot(pars);

	return 0;

}
