#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include "msHybridFunctions.h"

// Handles the user-defined inputs
msHybridArguments handleInputs(int argc, char* argv[])
{
	int i;
	msHybridArguments retval;
	
	/* Default setting */
	retval.IT_input = NULL;
	retval.FT_input = NULL;	
	retval.PLF_input = NULL;
	retval.hybrid_output = NULL;
	retval.scan_error = -1;
	retval.ppme = -1;
	retval.mode = 0;

	for (i=0; i<argc; i++) {
		// Ion Trap mzXML input file handling
		if (argv[i][0] == '-' && argv[i][1] == 'i') {
			retval.IT_input = &(argv[i][2]);
		}// if

		// Fourier Transform mzXML input file handling
		if (argv[i][0] == '-' && argv[i][1] == 'f') {
			retval.FT_input = &(argv[i][2]);
		}// if

		// PLF input file handling
		else if (argv[i][0] == '-' && argv[i][1] == 'p') {
			retval.PLF_input = &(argv[i][2]);
		}// else if
		
		// Hybrid output file handling
		else if (argv[i][0] == '-' && argv[i][1] == 'o') {
			retval.hybrid_output = &(argv[i][2]);
		}// else if				
		
		// Scan error window handling
		else if (argv[i][0] == '-' && argv[i][1] == 'l') {
			retval.scan_error = atoi(&(argv[i][2]));
		}// else if		

		// ppm error window handling
		else if (argv[i][0] == '-' && argv[i][1] == 'e') {
			retval.ppme = atoi(&argv[i][2]);
		}// else if	
	}// for

	// Input analysis
	if (!retval.IT_input) {
		printf("No ion trap data specified. Program exiting\n");
		exit(1);
	}// if	
	else if (!retval.FT_input) {
		printf("No FTICR data specified. Program exiting\n");
		exit(2);
	}// if
	else if (!retval.PLF_input) {
		printf("No piecewise lineair function specified. Program exiting\n");
		exit(3);
	}// else if
	else if (!retval.hybrid_output) {
		printf("No output file specified. Program exiting\n");
		exit(4);
	}// else if
	else if (retval.scan_error < 0) {
		printf("Scan search window not specified. Program exiting\n");
		exit(5);
	}// else if 
	else if (retval.ppme < 0) {
		printf("Mass error not specified. Program exiting\n");
		exit(6);
	}// else if

	return retval;

}// msWarpArguments handleInputs(int argc, char* argv[])


// Handles the PLF file and returns an array of PFL parts
PLF_Array handlePLF(char* PLF_file, int* mode)
{
	char line[1000];
	FILE* finput;
	float x1=0, x2=0, y1=0, y2=0;
	int alloc_size, read = 1;
	PLF_Array retval;

	// standard value assigning
	retval.function_array = malloc(sizeof(PLF_Function));
	retval.size = 0;
	alloc_size = 1;

	// Opening file and checking source
	finput = fopen(PLF_file, "r");
	if (!finput) {
		printf("PLF File could not be opened. Program exiting\n");
		exit(2);
	}// else if

	// First read the mode
	fgets(line, 999, finput);
	read = sscanf(line, "#mode %i", mode);

	// Then read the comment line 
	fgets(line, 999, finput);

	// First point scan	
	read = fscanf(finput, "%f\t%f", &x1, &y1);

	// Succeeding scans until no more scan
	while (read > 0) {		
		read = fscanf(finput, "%f\t%f", &x2, &y2);
		
		retval.function_array[retval.size].start_index = x1;
		retval.function_array[retval.size].end_index = x2;
		retval.function_array[retval.size].incline = ((y2-y1) / (x2-x1));
		retval.function_array[retval.size].offset = y2 - (x2 * retval.function_array[retval.size].incline);
		retval.size += 1;

		x1 = x2; y1 = y2;

		// Expanding array when needed
		if (retval.size == alloc_size) {
			alloc_size *= 2;
			retval.function_array = realloc(retval.function_array, (alloc_size * sizeof(PLF_Function)));
		}// if
	}// while

	// Sizing to the actual size
	retval.function_array = (PLF_Function*) realloc(retval.function_array, (retval.size * sizeof(PLF_Function)));

	return retval;

}// PLF_Array handlePLF(char* PLF_file)


// Returns the correct PLF_Function for the input
PLF_Function getPLF(double orig_val, PLF_Array plfa, int* found)
{
	int i;
	PLF_Function plff;

	for (i=0; i<plfa.size; i++) {
		plff = plfa.function_array[i];
		if (orig_val >= plff.start_index && orig_val < plff.end_index) {
			*found = 1;
			return plff;
		}// if
	}// for

	*found = 0;
	return plff;

}// PLF_Function getPLF(double index)


// Calculates the value for the plf and the x-input
double calcPLFValue(double orig_val, PLF_Function func)
{
	return (func.offset + (func.incline * orig_val));

}// double calcPLFValue(double index, PLF_Function func)


